//*************************************************************************************************
//
//  Description:
//    tyres_undeformable.fx - Non-deforming tyres shader with scrolling UVs - ideal for LODB tyres
//
//  <P> Copyright (c) 2007 Blimey! Games Ltd. All rights reserved.
//
//  Author: 
//    Tom Nettleship
//
//  History:
//
//  <TABLE>
//    \Author         Date        Version       Description
//    --------        -----       --------      ------------
//    TNettleship     06/13/2007  0.1           Created
//    TNettleship     07/11/2007  0.2           Changed lighting to work in world coords.
//    TNettleship     07/24/2007  0.3           Made sure samplers aren't using anisotropic filtering.
//    TNettleship     08/17/2007  0.4           Removed texture bias.
//    TNettleship     10/23/2007  0.5           Converted to on-load rendermode behaviour binding.
//    RDibley         18/01/2008  0.6           Copied tyres.fx and removed all bones specific stuff
//  <TABLE>
//
//*************************************************************************************************

#define _SSAO_READY_

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

#define MAX_ENVMAP_BIAS  6.0f


//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

#if defined( _3DSMAX_ )
float4x4 worldI : WorldInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;
#elif defined( ENV_MAPPING )
float4x4 envTransformI : EnvInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#endif



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool ProtectFromRescaling = true;
	bool export = false;
	bool use3dMapping = true;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif


//
// Simulation constants
//

float currentTime : TIME
<
> = 0.0f;

float textureRepeat															// Amount of times the tread texture repeats around the circumference
<
	string UIName = "Texture Repeat";
	bool appEdit = true;
> = 1.0f;

float angularVelocity														// Angular velocity of the wheel (in rad/s)
<
	string UIName = "Angular velocity";
	float UIMin = -1024.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 0.0f;

float axialRotation															// Current rotation of the wheel (in radians)
<
	string UIName = "Axial Rotation";
	float UIMin = -1024.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 0.0f;

float minBlurVelocity														// Angular velocity of minimum blur map
<
	string UIName = "Min Blur Vel";
	bool appEdit = true;
> = 0.0f;

float maxBlurVelocity														// Angular velocity of maximum blur map
<
	string UIName = "Max Blur Vel";
	bool appEdit = true;
> = 1.0f;

SPECIALISATION_PARAM( useDustMapping, "Use dust map?", "DUST_MAPPING" )		// True if the dust map is to be used.

#if defined( _3DSMAX_ ) || defined( DUST_MAPPING )
// Density of dust.
// Is tweakable in 3dsMax for preview of dust texture on the tyre,
// but at run-time, it is wired to physics/simulation value.
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( dustiness, "Dust Density", useDustMapping, 0.0f, 1.0f, 0.0f )
#endif	// DUST_MAPPING

//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap							// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE								// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;

texture specularTexture : TEXTURE								// Specular colour in RGB, shininess in alpha
<
	string UIName = "Spec Tex {UV1}";
	bool appEdit = true;
>;

texture normalTexture : TEXTURE									// Normal map
<
	string UIName = "Normal Tex {UV1}";
	bool appEdit = true;
>;

texture blurredDiffuseTexture : TEXTURE
<
	string UIName = "Blurred Diff {UV1}";
	bool appEdit = true;
>;

texture blurredSpecularTexture : TEXTURE
<
	string UIName = "Blurred Spec {UV1}";
	bool appEdit = true;
>;

texture blurredNormalTexture : TEXTURE					// Blurred normal map
<
	string UIName = "Blurred Norm {UV1}";
	bool appEdit = true;
>;

#if defined( _3DSMAX_ ) || defined( DUST_MAPPING )
DEPENDENT_TEXTURE_PARAM( dustTexture, "Dust Tex {UV1}", useDustMapping )		// Dust texture.
#endif	// DUST_MAPPING

SPECIALISATION_PARAM( useEnvironmentMap, "Use environment map?", "ENV_MAPPING" )	// TRUE if the environment map is to be used

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_CUBE_TEXTURE_PARAM( environmentTexture, "Env Texture", useEnvironmentMap, false )
#endif

float minSpecPower
<
	string UIName = "Min Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 1.0f;

float maxSpecPower
<
	string UIName = "Max Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 32.0f;

float globalSpecularFactor
<
	string UIName = "Specular Factor";
	float UIMin = 0.0f;
	float UIMax = 16.0f;
	bool appEdit = true;
> = 1.0f;

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_FLOAT_PARAM_MIN_MAX_DEFAULTED( globalEMapFactor, "EMap Factor", useEnvironmentMap, 0.0f, 16.0f, 1.0f )
#endif

//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS

// colour multiplier, forced to end to avoid artists touching it

float4 globalColourMultiplier
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = true;
> = { 1.0f, 1.0f, 1.0f, 1.0f };


//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


sampler2D blurredDiffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="blurredDiffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < blurredDiffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
 #endif
};

sampler2D blurredSpecularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="blurredSpecularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < blurredSpecularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
 #endif
};

sampler2D blurredNormalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="blurredNormalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
>
= sampler_state
{
	Texture = < blurredNormalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
 #endif
};


#if defined( _3DSMAX_ ) || defined( DUST_MAPPING )
sampler2D dustMap : SAMPLER
<
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="dustTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
>
= sampler_state
{
	Texture = < dustTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif	// DUST_MAPPING


#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
samplerCUBE environmentMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="environmentTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < environmentTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


//
// Functions
//


//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;                            // Object space position
#ifdef _3DSMAX_
	float4 colour   : TEXCOORD1;                          // Vertex colour
	float3 texCoord : TEXCOORD0;                          // UVW channel 1 texture coord - N.B. Max requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
#else
	float4 colour   : COLOR0;                              // Vertex colour
	float3 texCoord : TEXCOORD0;                          // UVW channel 1 texture coord
#endif

	float3 normal   : NORMAL;                              // Object space normal
	float3 tangent  : TANGENT;                            // Object space tangent
	float3 binormal : BINORMAL;                            // Object space normal
};


// Output structure
struct VSOUTPUT
{
	float4 position    : POSITION;                          // View-coords position
	float4 colour      : TEXCOORD5;                            // Vertex colour
	float2 texCoord    : TEXCOORD0;                        // UV coords for texture channel 0
	float3 normal      : TEXCOORD1;                        // Normal vector (world space)
	float4 eye        : TEXCOORD2;                        // Eye vector (world space)

	float3 tangent    : TEXCOORD3;                        // Tangent vector (world space)
	float3 binormal    : TEXCOORD4;                        // Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_VS( 6 )
};

struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD1;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
};


//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT TyresVertexShaderNoBones( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.colour = _input.colour;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space coordinate frame
	float3 normal     = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	_output.normal   = normal;
	_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
	_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	//
	// Animated vertex coords
	//

	// Get input w coord

	// If its very positive
	if ( _input.texCoord.z > 1.5f )
	{
		// Dont change the input UVs
		_output.texCoord.xy = _input.texCoord.xy;
	}

	else
	{
		// If its fairly positive
		if ( _input.texCoord.z > 0.5f )
		{
			// Rotate anticlockwise around centre
			float s, c;
#ifdef _3DSMAX_
			float test = ( currentTime / 5.0f ) * angularVelocity;
			sincos( -test, s, c );
#else
			sincos( -axialRotation, s, c );
#endif
			float2 rotatedCoords = float2( _input.texCoord.x, _input.texCoord.y );
			rotatedCoords -= 0.5f;
			rotatedCoords = float2( ( rotatedCoords.x * c ) + ( rotatedCoords.y * -s ),
															( rotatedCoords.x * s ) + ( rotatedCoords.y * c ) ) + 0.5f;
			_output.texCoord = rotatedCoords;
		}

		else
		{
			// If its very negative
			if ( _input.texCoord.z < -0.5f )
			{
				// Rotate clockwise around centre
				float s, c;
#ifdef _3DSMAX_
				float test = ( currentTime / 5.0f ) * angularVelocity;
				sincos( test, s, c );
#else
				sincos( axialRotation, s, c );
#endif
				float2 rotatedCoords = float2( _input.texCoord.x, _input.texCoord.y );
				rotatedCoords -= 0.5f;
				rotatedCoords = float2( ( rotatedCoords.x * c ) + ( rotatedCoords.y * -s ),
																( rotatedCoords.x * s ) + ( rotatedCoords.y * c ) ) + 0.5f;
				_output.texCoord = rotatedCoords;
			}

			// Otherwise (its near 0)
			else
			{
				// Scroll in U direction
#ifdef _3DSMAX_
				float test = ( currentTime / 5.0f ) * angularVelocity;
				_output.texCoord.x = _input.texCoord.x + ( ( test * textureRepeat ) / ( 2.0f * 3.14159265f ) );
#else
				_output.texCoord.x = _input.texCoord.x + ( ( axialRotation * textureRepeat ) / ( 2.0f * 3.14159265f ) );
#endif
				_output.texCoord.y = _input.texCoord.y;
			}
		}
	}

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	_output.colour *= globalColourMultiplier;

	return _output;
}



VSOUTPUT_LD TyresNoBonesLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.colour = _input.colour;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate world-space coordinate frame
	float3 normal = normalize( mul( float4( _input.position, 1.0f ), world ).xyz );
	
	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	//
	// Animated vertex coords
	//

	// Get input w coord

	// If its very positive
	if ( _input.texCoord.z > 1.5f )
	{
		// Dont change the input UVs
		_output.texCoord.xy = _input.texCoord.xy;
	}
	else
	{
		// If its fairly positive
		if ( _input.texCoord.z > 0.5f )
		{
			// Rotate anticlockwise around centre
			float s, c;
#ifdef _3DSMAX_
			float test = ( currentTime / 5.0f ) * angularVelocity;
			sincos( -test, s, c );
#else
			sincos( -axialRotation, s, c );
#endif
			float2 rotatedCoords = float2( _input.texCoord.x, _input.texCoord.y );
			rotatedCoords -= 0.5f;
			rotatedCoords = float2( ( rotatedCoords.x * c ) + ( rotatedCoords.y * -s ),
															( rotatedCoords.x * s ) + ( rotatedCoords.y * c ) ) + 0.5f;
			_output.texCoord = rotatedCoords;
		}

		else
		{
			// If its very negative
			if ( _input.texCoord.z < -0.5f )
			{
				// Rotate clockwise around centre
				float s, c;
#ifdef _3DSMAX_
				float test = ( currentTime / 5.0f ) * angularVelocity;
				sincos( test, s, c );
#else
				sincos( axialRotation, s, c );
#endif
				float2 rotatedCoords = float2( _input.texCoord.x, _input.texCoord.y );
				rotatedCoords -= 0.5f;
				rotatedCoords = float2( ( rotatedCoords.x * c ) + ( rotatedCoords.y * -s ),
																( rotatedCoords.x * s ) + ( rotatedCoords.y * c ) ) + 0.5f;
				_output.texCoord = rotatedCoords;
			}

			// Otherwise (its near 0)
			else
			{
				// Scroll in U direction
#ifdef _3DSMAX_
				float test = ( currentTime / 5.0f ) * angularVelocity;
				_output.texCoord.x = _input.texCoord.x + ( ( test * textureRepeat ) / ( 2.0f * 3.14159265f ) );
#else
				_output.texCoord.x = _input.texCoord.x + ( ( axialRotation * textureRepeat ) / ( 2.0f * 3.14159265f ) );
#endif
				_output.texCoord.y = _input.texCoord.y;
			}
		}
	}

	// Do vertex lighting
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	// keep alpha value from the vertex
	_output.colour.a = _input.colour.a;

	return _output;
}






//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

// Input structure
struct PSINPUT
{
	float4 colour   : TEXCOORD5;					// Vertex colour
	float2 texCoord : TEXCOORD0;			// UV coords for texture channel 0
	float3 normal   : TEXCOORD1;			// Normal vector (world space)
	float4 eye      : TEXCOORD2;			// Eye vector (world space)
	float3 tangent	: TEXCOORD3;			// Tangent vector (world space)
	float3 binormal : TEXCOORD4;			// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 6 )
};

#else

struct PSINPUT
{
	float4 colour      : TEXCOORD5;									// Vertex colour
	float2 texCoord    : TEXCOORD0;								// UV coords for texture channel 0
	float3 normal      : TEXCOORD1_centroid;			// Normal vector (world space)
	float4 eye				 : TEXCOORD2_centroid;			// Eye vector (world space)

	float3 tangent     : TEXCOORD3_centroid;			// Tangent vector (world space)
	float3 binormal    : TEXCOORD4_centroid;			// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 6 )
	DECLARE_SHADOW_PS_INPUTS
};

#endif


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT TyresFragmentShaderNoBones( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	float globalSpecularFactorValue = globalSpecularFactor;
	float minSpecPowerValue = minSpecPower;
	float maxSpecPowerValue = maxSpecPower;

	// Read textures
	float4 unblurredDiffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	float4 unblurredSpecularTexColour = tex2D( specularMap, _input.texCoord );
	float4 blurredDiffuseTexColour = tex2D( blurredDiffuseMap, _input.texCoord );
	float4 blurredSpecularTexColour = tex2D( blurredSpecularMap, _input.texCoord );
	// Note: Ideally, we would do fetches for dust mapping here, too, to match with existing structure.
	// As declaring variables conflicts with dependent code, all the dust related computation,
	// including fetches, is moved down, near the location where it is really needed.

	// Normalise interpolated vectors
	float3 TSnormal = normalize( _input.normal );
	float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// Normalise the input tangent and binormal vectors
	float3 tangent = normalize( _input.tangent );
	float3 binormal = normalize( _input.binormal );

	// Fetch and decode the map normals
	float4 normalMapColour = tex2D( normalMap, _input.texCoord );
	float4 blurredNormalMapColour = tex2D( blurredNormalMap, _input.texCoord );
	float3 unblurredNormalFromMap = ( normalMapColour.rgb * 2.0f ) - 1.0f;
	float3 blurredNormalFromMap = ( blurredNormalMapColour.rgb * 2.0f ) - 1.0f;

	// Interpolate between the unblurred and blurred diffuse colours and normals
//  float interpolant = smoothstep( minBlurVelocity, maxBlurVelocity, abs( angularVelocity ) );
	float interpolant = clamp( abs( angularVelocity ), minBlurVelocity, maxBlurVelocity );
	interpolant = ( interpolant - minBlurVelocity ) / ( maxBlurVelocity - minBlurVelocity );

	float4 diffuseTexColour = lerp( unblurredDiffuseTexColour, blurredDiffuseTexColour, interpolant );
	float4 specularTexColour = lerp( unblurredSpecularTexColour, blurredSpecularTexColour, interpolant );
	float3 normalFromMap = normalize( lerp( unblurredNormalFromMap, blurredNormalFromMap, interpolant ) );

	// Perturb the tangent space normal by the normal map
	normal = ( normalFromMap.z * TSnormal ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
	normal = normalize( normal );

	DEPENDENT_CODE_START( useDustMapping )
#if defined( _3DSMAX_ ) || defined( DUST_MAPPING )
	float4 unblurredDustTexColour = tex2D( dustMap, _input.texCoord );
	// Instead of explicit pre-blurred dust texture, we use lower-level mipmap - skipping one level.
	float4 blurredDustTexColour = tex2Dbias( dustMap, float4( _input.texCoord, 2.0f, 2.0f ) );

	float4 dustTexColour = lerp( unblurredDustTexColour, blurredDustTexColour, interpolant );

	// Layer the dust onto main material, in a blend-first way.
	// Unfortunately, we can not perform blend-last (due to insufficient support in lighting code),
	// Density of dust is taken from alpha channel of dust map, and adjusted by overall dustiness.
	// The more dust there is, the less specularity / reflectivity will the fragment exhibit.
	float dustFactor = ( dustiness * dustTexColour.a );
	diffuseTexColour.rgb = lerp( diffuseTexColour.rgb, dustTexColour.rgb, dustFactor );
	specularTexColour.rgb = specularTexColour.rgb * ( 1.0f - dustFactor );
#endif	// DUST_MAPPING
	DEPENDENT_CODE_END( useDustMapping )

#if defined( PSEUDO_GI )

	// If environment mapping is switched on
	float4 accumulatedColour;
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
		// Calculate ambient-lit base colour (looking up the envmap for the ambient lighting)
		float4 envmapambient = texCUBElod( environmentMap, float4( normal.xyz, 4 ) );
		accumulatedColour = diffuseTexColour * envmapambient;
#endif
	DEPENDENT_CODE_ELSE( useEnvironmentMap )
#if defined( _3DSMAX_ ) || !defined( ENV_MAPPING )
		// Calculate ambient-lit base colour from interpolated lighting
		accumulatedColour = diffuseTexColour * _input.colour;
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

#else
	float4 accumulatedColour = diffuseTexColour * _input.colour;
#endif

	// If environment mapping is switched on
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
		// Calculate the envmap space reflection vector
#if defined( _3DSMAX_ )
		// Max uses a map aligned to the object, always
		float3 envMapSpaceNormal = mul( float4( normal, 0.0f ), worldI ).xyz;
		float3 envMapSpaceEye = mul( float4( -eye, 0.0f ), worldI ).xyz;
#else
		// The engine uses a map aligned to its own space, which may be object aligned, or to a parent object
		float3 envMapSpaceNormal = mul( float4( normal, 0.0f ), envTransformI ).xyz;
		float3 envMapSpaceEye = mul( float4( -eye, 0.0f ), envTransformI ).xyz;
#endif
		float3 reflectionVector = reflect( envMapSpaceEye, envMapSpaceNormal );

		// Fetch the environment map colour using the world coords vector
#ifdef _3DSMAX_
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector.xzy, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#else
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#endif

		// Calculate envmap colour (back in tangent space is fine) and add to diffuse
		accumulatedColour += specularTexColour * environmentTexColour * globalEMapFactor;
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = _input.colour.w; // enforce no alpha tyres diffuseTexColour.w * _input.colour.w;
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}



PSOUTPUT TyresNoBonesLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	DEPENDENT_CODE_START( useDustMapping )
#if defined( _3DSMAX_ ) || defined( DUST_MAPPING )
	float4 dustTexColour = tex2D( dustMap, _input.texCoord );
	float dustFactor = ( dustiness * dustTexColour.a );
	diffuseTexColour.rgb = lerp( diffuseTexColour.rgb, dustTexColour.rgb, dustFactor );
#endif	// DUST_MAPPING
	DEPENDENT_CODE_END( useDustMapping )

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

	// Apply vertex lighting
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateLowDetailOutputPixel( accumulatedColour );

	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Tyres_NoBones
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;

	string normalBehaviour    = "ERMB_RENDER";
	string normalTechnique    = "Tyres_NoBones";
	int    normalDeferredID    = 0;

	string zprimeBehaviour    = "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";

	string shadowGenBehaviour  = "ERMB_RENDER_DEFAULT";
	int    shadowGenDeferredID = 0;

	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Tyres_NoBones_LowDetail";
	int    lowDetailDeferredID = 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx TyresVertexShaderNoBones();
		PixelShader = compile sce_fp_rsx TyresFragmentShaderNoBones();
#else
		VertexShader = compile vs_3_0 TyresVertexShaderNoBones();
		PixelShader = compile ps_3_0 TyresFragmentShaderNoBones();
#endif
	}
}


technique Tyres_NoBones_LowDetail
<
	bool preservesGlobalState = true;

	string normalBehaviour    = "ERMB_RENDER";
	string normalTechnique    = "Tyres_NoBones_LowDetail";
	int    normalDeferredID    = 0;

	string zprimeBehaviour    = "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour  = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx TyresNoBonesLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx TyresNoBonesLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 TyresNoBonesLowDetailVertexShader();
		PixelShader = compile ps_3_0 TyresNoBonesLowDetailFragmentShader();
#endif
	}
}
